import java.util.Random;
import java.util.ArrayList;
import java.io.File;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;

/**
 * Die Klasse DATAHANDLER stellt 3 Methoden bereit, um Daten für den k-Means-Algorithmus zu initalisieren
 * 
 * @author Daniela Andres
 */
public class DATAHANDLER {

    /**
     * Eine Arraylist, die alle Points des Datensatzes hält
     */
    ArrayList<POINT> pointList;

    /**
     * Obere Schranke für die x- und y-Werte der in pointList enthaltenen Punkte
     */
    int maxValue;
    /**
     * Untere Schranke für die x- und y-Werte der in pointList enthaltenen Punkte
     */
    int minValue;
    /**
     * Anzahl der Datenpunkte = Länge der pointList
     */
    int count;

    public DATAHANDLER()
    {
        pointList = new ArrayList<>();
        getRandomCluster(4,300,20,-20);
    }

    /**
     * getRandomData gibt countPoint-viele zufällige Datenpunkte zwischen min und max zurück
     * 
     * @param countPoint
     *        Anzahl der zu erstellenden Datenpunkte 
     * @param max
     *        Gewünschter maximaler Wert der x- und y-Koordinaten der Datenpunkte
     * @param min
     *        Gewünschter minimaler Wert der x- und y-Koordinaten der Datenpunkte
     */
    public void getRandomData(int countPoint, int max, int min){
        pointList = new ArrayList<POINT>();
        
        while(pointList.size()<countPoint){
            double dataX = (Math.random() * (max - min)) + min;
            double dataY = (Math.random() * (max - min)) + min;
            POINT p = new POINT(dataX, dataY);
            if(!pointList.contains(p)){
                pointList.add(p);
            }
        }

        maxValue = max;
        minValue = min;
        count = pointList.size();
    }

    /**
     * getRandomCluster gibt k-viele Cluster mit insgesamt anzahl Datenpunkten zurück. Max und min definieren
     * dabei den Bereich, in dem die Clusterzentren liegen. Aufgrund der zufälligen Streuung sind die 
     * tatsächlichen Maximal- und Minmalwerte der Datenpunkte anders als die eingetragenen Werte für
     * Max und Min
     * 
     * @param countCluster
     *        Anzahl der gewünschten Cluster
     * @param countPoint
     *        Anzahl der zu erstellenden Datenpunkte 
     * @param max
     *        Gewünschter maximaler Wert der Clusterzentren
     * @param min
     *        Gewünschter minimaler Wert der Clusterzentren
     */
    public void getRandomCluster(int countCluster, int countPoint, int max, int min){
        pointList = new ArrayList<POINT>();
        var centers = new ArrayList<POINT>();

        outer:
        while(centers.size() < countCluster){
            double dataX = (Math.random() * (double) (max - min)) + min;
            double dataY = (Math.random() * (double) (max - min)) + min;
            POINT p = new POINT(dataX, dataY);

            for(int i = 0; i < centers.size(); i++){
                if(p.getDistance( centers.get(i)) < (double)(max-min)/(Math.max(2, countCluster-1))){
                    continue  outer;
                }
            }
            centers.add(p);

        }

        for(int i = 0; i < countPoint; i++){
            //zufälliges Zentrum wählen
            Random r = new Random();
            int index = r.nextInt(centers.size());
            double dataX = 0;
            double dataY = 0;
            //Mit einer kleinen Wahrscheinlichkeit vollkommen zufälligen Punkt setzen
            if(Math.random() < 0.05){
                dataX = (Math.random() * (double) (max - min)) + min;
                dataY = (Math.random() * (double) (max - min)) + min;
            }else{
                //Sonst einen Punkt in einem zufälligen Radius um das ausgewählte Zentrum setzen 
                double radius = (Math.random() * (double) (max - min)/countCluster);
                dataX = centers.get(index).getX() + (2.0 * Math.random() - 1) * radius;
                dataY = centers.get(index).getY() + (2.0 * Math.random() - 1) * radius;
                POINT p = new POINT(dataX, dataY);
                pointList.add(p);
            }
        }

        //Max und Min Value abhängig von verschobenen Points --> anpassen
        double maxV = Double.MIN_VALUE;
        double minV = Double.MAX_VALUE;
        for (int i = 0; i < pointList.size(); i++) {
            double x = pointList.get(i).getX();
            double y = pointList.get(i).getY();
            if (x > maxV) {
                maxV = x;
            }if (y > maxV) {
                maxV = y;
            }if (x < minV) {
                minV = x;
            }if (y < minV) {
                minV = y;
            }
        }
        maxValue = (int) Math.ceil(maxV);
        minValue = (int) Math.floor(minV);
        count = countPoint;
    }

    /**
     * loadData kann eine csv-Datei einlesen, wenn sie in dem Ordner "Beispieldaten" des BlueJ-Projektes liegt
     * Dafür nötiges Format dieser csv-Datei: 
     *      <AchsennameX>;<AchsennameY>
     *      <X-Wert>;<Y-Wert>
     *      <X-Wert>;<Y-Wert>
     *      ...
     *      
     * Ein Beispiel für eine gültige Datei wäre:
     *      Zeit;Geschwindigkeit
     *      0.5;5
     *      1.0;10.3
     *      
     * @param fileName
     *        Name der Datei. Es muss der genaue Dateiname (mit Dateiendung!) verwendet werden. 
     *        Bei Rechtschreib- und Tippfehlern kann die Datei nicht gefunden werden. 
     *        Eingabebeispiel: "twoMoons.csv"
     *        
     */    
    public void loadData(String fileName){

        File file = new File("Beispieldaten\\" + fileName);

        if (!file.canRead() || !file.isFile()){
            System.out.println("File nicht gefunden.");
            System.exit(0);
        }

        BufferedReader reader = null;
        try {
            pointList.clear();

            reader = new BufferedReader(new FileReader("Beispieldaten/" + fileName));
            String zeile = null;
            //TODO: Koordinatenachsennamen rauslesen
            String achsennamen = reader.readLine();
            while ((zeile = reader.readLine()) != null) {
                String[] pointBuffer = zeile.split(";");
                pointList.add(new POINT(Double.parseDouble(pointBuffer[0]), Double.parseDouble(pointBuffer[1])));
            }

            //count, minValue und maxValue setzen
            double maxV = Double.MIN_VALUE;
            double minV = Double.MAX_VALUE;
            for (int i = 0; i < pointList.size(); i++) {
                double x = pointList.get(i).getX();
                double y = pointList.get(i).getY();
                if (x > maxV) {
                    maxV = x;
                }if (y > maxV) {
                    maxV = y;
                }if (x < minV) {
                    minV = x;
                }if (y < minV) {
                    minV = y;
                }
            }
            maxValue = (int) Math.ceil(maxV);
            minValue = (int) Math.floor(minV);
            count = pointList.size();

        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (reader != null)
                try {
                    reader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
        } 

    }
    
    /**
     * Umwandlung der ArrayList in ein Array, da die Klasse KMEANS mit Arrays arbeitet.
     */
    public POINT[] getPointlist(){
        POINT[] res = new POINT[pointList.size()];
        for(int i = 0; i < res.length; i++){
            res[i] = pointList.get(i);
        }
        return res; 
    }

}
