package jpp.tv;

import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Repraesentiert eine einzelne Sendung.
 * 
 * @author Andreas Bauer, a.bauer@uni-wuerzburg.de
 * 
 */
public class Telecast {

	private String name;
	private Channel channel;
	private Date start;
	private Date end;

	/**
	 * Der Konstruktor der Klasse Telecast.
	 * 
	 * @param name
	 *            Der Name der Sendung.
	 * @param channel
	 *            Der Kanal, auf dem die Sendung laeuft.
	 * @param start
	 *            Der Startzeitpunkt der Sendung.
	 * @param end
	 *            Der Endzeitpunkt der Sendung.
	 * @throws NullPointerException
	 *             Wird geworfen, wenn einer der uebergebenen Parameter null
	 *             ist.
	 * @throws IllegalArgumentException
	 *             Wird geworfen, wenn der uebergebene Startzeitpunkt zeitlich
	 *             hinter dem uebergebenen Endzeitpunkt liegt.
	 */
	public Telecast(String name, Channel channel, Date start, Date end)
			throws NullPointerException, IllegalArgumentException {
		if (name == null || channel == null || start == null || end == null)
			throw new NullPointerException(
					"Keiner der im vier Konstruktor uebergebenen Parameter darf null sein.");
		if (!start.before(end))
			throw new IllegalArgumentException(
					"Der Startzeitpunkt, der dem Konstruktor uebergeben wird, darf nicht zeitlich hinter dem uebergebenen Endzeitpunkt liegen.");
		this.name = name;
		this.channel = channel;
		this.start = start;
		this.end = end;
	}

	/**
	 * Gibt den Namen der Sendung als String zurueck.
	 * 
	 * @return Der Name der Sendung als String.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Gibt den Kanal der Sendung als Channel zurueck.
	 * 
	 * @return Der Kanal der Sendung als Channel.
	 */
	public Channel getChannel() {
		return channel;
	}

	/**
	 * Gibt den Startzeitpunkt der Sendung als java.util.Date zurueck.
	 * 
	 * @return Der Startzeitpunkt der Sendung als java.util.Date zurueck.
	 */
	public Date getStart() {
		return start;
	}

	/**
	 * Gibt den Endzeitpunkt der Sendung als java.util.Date zurueck.
	 * 
	 * @return Der Endzeitpunkt der Sendung als java.util.Date zurueck.
	 */
	public Date getEnd() {
		return end;
	}

	/**
	 * Gibt eine Stringrepraesentation der Sendung im Format
	 * "channel name: start - end" zurueck.
	 * 
	 * @return Eine Stringrepraesentation der Sendung im Format
	 *         "channel name: start - end"
	 */
	public String toString() {
		String result = "";
		SimpleDateFormat s = new SimpleDateFormat("dd.MM.yyyy HH:mm");
		result = this.channel + " " + this.name + ": " + s.format(this.start)
				+ " - " + s.format(this.end);
		return result;
	}

	/**
	 * Vergleicht das als Parameter uebergebene Objekt o mit diesem Objekt. Ist o ein Objekt der Klasse Telecast und stimmt in allen vier Attributen mit diesem Objekt ueberein, wird true zurueckgegeben, false sonst.
	 * 
	 * @param Das Objekt, mit dem dieses Objekt verglichen werden soll.
	 * @return Stimmt das als Parameter uebergebene Objekt in allen vier Attributen mit diesem Objekt ueberein, wird true zurueckgegeben, false sonst.
	 */
	public boolean equals(Object o) {
		if (o instanceof Telecast) {
			Telecast t = (Telecast) o;
			if (t.getChannel().equals(this.channel)
					&& t.getName().equals(this.name)
					&& t.getEnd().equals(this.end)
					&& t.getStart().equals(this.start))
				return true;
		}
		return false;
	}
	
	/**
	 * Ueberlaedt die in java.lang.Object definierte hashCode-Methode. Berechnet den Hash-Code nach folgendem Verfahren: hashCode = 31 * (31 * (31 * (31 + channel.hashCode()) + name.hashCode()) + start.hashCode()) + end.hashCode() 
	 * 
	 * @return Der hashCode fuer dieses Objekt der Klasse Telecast.
	 */
	public int hashCode(){
		return 31 * (31 * (31 * (31 + channel.hashCode()) + name.hashCode()) + start.hashCode()) + end.hashCode(); 
	}
	
}
